/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package jdbc;

import java.io.FileOutputStream;

import com.inet.report.*;

/**
 * This sample class shows how to create a report that is based on a stored procedure with two input parameters. The
 * vital method calls are Datasource.createTableSource(String) and TableSource.refresh(). These methods creating a
 * TableSource object for the stored procedure and add DatabaseFields and PromptFields to the report. To execute this
 * sample the following is necessary: - an installed JDBC driver for MS SQL Server (e.g. i-net SERO, bundled with i-net
 * Clear Reports) - the MS SQL Server northwind database - Data Source Configuration for the northwind database, created
 * with Data Source Manager in i-net Designer or Configuration Manager
 */
public class ReportBasedOnStoredProcedure {

    /**
     * main method of the sample, showing how to create a report based on a stored procedure
     * @param args not needed
     */
    public static void main( String[] args ) {
        try {
            //create a new report
            Engine reportEngine = RDC.createEmptyEngine( Engine.EXPORT_PDF );

            //define the datasource. (in this sample it is a northwind database on the MS SQL Server)
            Datasource ds = reportEngine.getDatabaseTables().createDatasource( "<dataSourceConfigurationName>" ); // e.g. Northwind

            // set the password if it is not saved in the Data Source Configuration
            ds.setPassword( "<Password>" );

            //!!!  define a SQL statement as base of the report 
            TableSource sp_salesByYear = ds.createTableSource( "Sales by Year" );

            //create PrompTFields for SP and set input parameter
            //using this function is only necessary if SP needs special values for execution
            //please note the parameter name pattern depends on database, for the MS SQL the names should begins with '@'. 
            //sp_salesByYear.setInputParameter_TypesOfSQLType(new String[]{"Beginning_Date","Ending_Date"},new int[]{Types.DATE,Types.DATE},new Object[]{new Date(90,0,1),new Date(100,0,1)});

            //add columns to TableSource. Add PromptFields for Beginning_Date and Ending_Date SP parameter 
            sp_salesByYear.refresh();

            //put database fields in the report
            Area dArea = reportEngine.getArea( "D" );
            Section detailSection = dArea.getSection( 0 );
            DatabaseField fieldShippedDate = sp_salesByYear.getDatabaseField( sp_salesByYear.getColumnName( 1 ) );
            DatabaseField fieldOrderID = sp_salesByYear.getDatabaseField( sp_salesByYear.getColumnName( 2 ) );
            DatabaseField fieldOrder_Subtotals = sp_salesByYear.getDatabaseField( sp_salesByYear.getColumnName( 3 ) );

            detailSection.setHeight( 1100 );
            detailSection.addFieldElement( fieldOrderID, 50, 50, 2000, 100 );
            detailSection.addFieldElement( fieldOrder_Subtotals, 2100, 50, 1000, 1000 );
            detailSection.addFieldElement( fieldShippedDate, 3200, 50, 2000, 100 );

            //generate the pdf document
            reportEngine.execute();

            //put the document on harddisk
            int pageCount = reportEngine.getPageCount();
            FileOutputStream fos = new FileOutputStream( "c:/test.pdf" );
            for( int i = 0; i < pageCount; i++ ) {
                fos.write( reportEngine.getPageData( i + 1 ) );
            }
            fos.flush();
            fos.close();
            System.exit( 0 );

        } catch( ReportException re ) {
            System.err.println( "generating report failed" );
            re.printStackTrace();
        } catch( Exception e ) {
            System.err.println( "creating pdf file failed" );
            e.printStackTrace();
        }
    }
}
